// PID_sum_terms.v
//
//	Module for computing the sum of the P-term, I-term, and D-term
//		outputs to produce the total PID output.
//
`timescale 1ps/1fs

module PID_sum_terms(
    output reg signed [19:0] out,
    input clk,				// Clock signal for updating the PID output
    input wire signed [19:0] P_out,	// P-term output, a 20-bit signed integer
    input wire signed [23:0] I_out,	// I-term output, a 24-bit signed integer
    input wire signed [23:0] D_out );	// D-term output, a 24-bit signed integer

    parameter delay = 20;
    parameter initial_duty = 0.25195;
    integer max_count = 2048;
    integer i_tmp;

    initial begin
	out = $rtoi( initial_duty * max_count + 0.5 ); 
	$monitor( "Time: %8d, initial_duty=%.6e, out=%d", $time, initial_duty, out );
    end

    always @ ( posedge clk ) begin
	// iKP, iKI, and iKD are the integral values of (16 * KP),
	// (2048 * KI), and (16 * KD), respectively.  First, produce a sum
	// of P_out + (I_out / 128) + D_out.
	i_tmp = P_out + (I_out >>> 7);

	// Limit (P_out + (I_out / 128)) to a 24-bit signed integer.  This
	//   step is not necessary but it is used so that it is performing
	//   the same limiting process as the one employed by the DPWM model
	//   using the adv. digital components.
	if ( i_tmp > 8388607 )
	    i_tmp = 8388607;
	else if ( i_tmp < -8388608 )
	    i_tmp = -8388608;

	// Now carry out the addition of D_out.
	i_tmp = i_tmp + D_out;

	// Limit the result of the previous addition to a 24-bit signed
	//   integer.  This step is not necessary but it is used so that it
	//   is performing the same limiting process as the one employed by
	//   the DPWM model using the adv. digital components.
	if ( i_tmp > 8388607 )
	    i_tmp = 8388607;
	else if ( i_tmp < -8388608 )
	    i_tmp = -8388608;


	// The delay was purposely halved so an adv. digital buffer with a
	//   delay of (delay/2) is put in to follow the output of the PID.
	//   Also, the final division by 16 is carried out here.
	out <= #(delay/2) (i_tmp >>> 4);
    end

endmodule
