// I_term.v
//
//	Module for computing the I-term output of the PID filter.
//		The integration employs trapezoidal rule.  Hence,
//		out(n) = out(n - 1) + iKI * (err(n) + err(n-1)) / 2
//
`timescale 1ps/1fs

module I_term(
    output reg signed [23:0] out,	// I-term output, a 24-bit signed integer
    input clk,				// Clock signal to produce the I-term output
    input wire signed [9:0]  err_n0,	// Current error e(n), a 10-bit signed integer
    input wire signed [9:0]  err_n1,	// Previous error e(n-1), a 10-bit signed integer
    input wire signed [9:0]  iKI ) ;	// 2048 * KI stored as a 10-bit signed integer

    parameter delay = 20;
    parameter initial_duty = 0.25195;
    integer max_count = 2048;
    integer i_tmp;

    initial begin
	out = $rtoi( initial_duty * max_count + 0.5 ); 
	// But iKI is actually the integer value of (2048 * KI).  So after
	// computing the normal initial output, multiply it by 2048.
	out = out * 2048;
    end

    always @ ( posedge clk  ) begin
	// Compute (err_n0 + err_n1).  Limit the result to a 10-bit
	//   signed integer.
	i_tmp = err_n0 + err_n1;
	if ( i_tmp > 511 )
	    i_tmp = 511;
	else if ( i_tmp < -512 )
	    i_tmp = -512;

	// Compute iKI * (err_n0 + err_n1)
	i_tmp = iKI * i_tmp;

	// Compute out(n-1) + iKI * (err_n0 + err_n1) / 2.  Limit the result
	//   to a 24-bit signed integer.
	i_tmp = out + (i_tmp >>> 1);
	if ( i_tmp > 8388607 )
	    i_tmp = 8388607;
	else if ( i_tmp < -8388608 )
	    i_tmp = -8388608;

	// Assign the output
	out = #(delay) i_tmp;
    end

endmodule
