// D_term.v
//
//	Module for computing the D-term output of the PID filter.  The
//		actual transfer function includes an additional low-pass
//		filter.  The total transfer function implemented is
//
//		OUT(z) = Tlp(z) * KD * [1 - z^-1] * E(z)
//		Tlp(z) = z / (2 * z - 1)
//
`timescale 1ps/1fs

module D_term(
    output reg signed [23:0] out,	// D-term output, a 24-bit signed integer
    input clk,				// Clock signal to produce the D-term output
    input wire signed [9:0]  err_n0,	// Current error, a 10-bit signed integer
    input wire signed [9:0]  err_n1,	// Previous error, a 10-bit signed integer
    input wire signed [13:0] iKD ) ;	// 16 * KD stored as a 14-bit signed integer

    parameter delay = 20;
    integer i_tmp;

    initial begin
	out = 0;
    end

    always @ ( posedge clk  ) begin
	// Compute (err_n0 - err_n1).  Limit the result to a 6-bit
	//   signed integer.
	i_tmp = err_n0 - err_n1;
	if ( i_tmp > 31 )
	    i_tmp = 31;
	else if ( i_tmp < -32 )
	    i_tmp = -32;

	// Compute (KD * (err_n0 - err_n1))
	i_tmp = iKD * i_tmp;


	// Compute the low-pass response of Tlp(z) = z / (2 * z - 1).  Hence,
	//   OUT(z)/IN(z) = z / (2 * z - 1)
	//   OUT(z) * (2 * z - 1) = z * IN(z)
	//   OUT(z) * (1 - 0.5 * z^-1) = 0.5 * IN(z)
	//   out(n) - 0.5 * out(n-1) = 0.5 * in(n)
	//   out(n) = 0.5 * [ in(n) + out(n-1) ]

	// in(n) + out(n-1)
	i_tmp = out + i_tmp;

	// Limit in(n) + out(n-1) to a 24-bit signed integer.
	if ( i_tmp > 8388607 )
	    i_tmp = 8388607;
	else if ( i_tmp < -8388608 )
	    i_tmp = -8388608;

	// Assign (i_tmp / 2) to the new out
	out = #(delay) (i_tmp >>> 1);
    end

endmodule
